/*-----------------------------------------------------------------------------

   QUASAR - q-gram Alignment based on Suffix ARrays

   Copyright (C) 1998 Stefan Burkhardt
   Author: Stefan Burkhardt <stburk@mpi-sb.mpg.de>
   This file is part of the QUASAR package.

   QUASAR is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   QUASAR is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the QUASAR package; see the file copying.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  or contact the author. 

-------------------------------------------------------------------------------

  ncbi blast interface
  
  $File$
  $Revision: 1.16 $
  $Date: Wed, 29 Mar 2000 11:07:45 +0200 $

-----------------------------------------------------------------------------*/

#include "q_blast.h"
#include "ncbi.h"
#include "objseq.h"
#include "objsset.h"
#include "sequtil.h"
#include "seqport.h"
#include "tofasta.h"
#include "blast.h"
#include "blastpri.h"
#include "simutil.h"
#include "txalign.h"
#include "gapxdrop.h"
#include "sqnutils.h"
#include "tofasta.h"
#include "lookup.h"
#include "objcode.h"
#include "readdb.h"
#include "ncbithr.h"
#include "dust.h"

int	**memdb;

NlmMFILEPtr LIBCALL MyNlmOpenMFILE (int	*name)
{
  NlmMFILEPtr mfp;

  if ((mfp=(NlmMFILEPtr) MemNew(sizeof(NlmMFILE))) == NULL)
    return NULL;
  
  mfp->mem_mapp = (Nlm_MemMapPtr)Nlm_MemNew(sizeof(Nlm_MemMap));
  if(mfp->mem_mapp == NULL)
    return NULL;
  mfp->mem_mapp->file_size = (Nlm_Int4) name[0] * sizeof(int);
  mfp->mem_mapp->mmp_begin = (Uint1Ptr) name+sizeof(int);
  
  /* copy this pointer to where it's convenient. */
  mfp->mmp_begin = mfp->mmp = (Uint1Ptr) mfp->mem_mapp->mmp_begin;
  mfp->mfile_true = TRUE;
  mfp->mmp_end = mfp->mmp_begin + mfp->mem_mapp->file_size;
  mfp->contents_allocated = TRUE;
  
  return mfp;
}	/* MyNlmOpenMFILE */


#include "q_blastfns.c"
#include "q_readdb.c"

#define DEFLINE_BUF 255

/* Here we simulate argc and argv */
int	argc = 9;
char	argvec[] = "blastall\0-p\0blastn\0-i\0blast_query\0-d\0filtered_db\0-o\0quasar_hitsqddddddddddddddddd\0";
char	**argv;
int	setup_done = 0;
int	last_seq;     /* So we dont print one sequence header more than once */

static int LIBCALLBACK
tick_callback(Int4 sequence_number, Int4 number_of_positive_hits)
{
  return 0;
}

void MyWriteScore(ScorePtr score, FILE *outfp)
  /*  typedef struct score {
      ObjectIdPtr id;
      Uint1 choice;		 0=not set, 1=int, 2=real
      DataVal value;
      struct score PNTR next;    for sets of scores 
      } Score, PNTR ScorePtr;
  */
{
  int		score_written = 0;
  double	bit_score_written = 0.0;
  ScorePtr current = score;

  if(current == NULL)
    printf("NO SCORE!!!!\n");
  while(current != NULL)
    {
      /* Irrelevant entries
	 printf("score->id->id:  %d \n", current->id->id);
	 printf("score->id->str: %s \n", current->id->str);
	 printf("score->choice:  %d \n", current->choice);
	 printf("score->value.ptrvalue:   %d \n", current->value.ptrvalue);
	 printf("score->value.boolvalue:  %d \n", current->value.boolvalue);
	 printf("score->value.funcvalue:  %d \n", current->value.funcvalue);
	 printf("score->next:    %d \n", current->next);
      */
      if(strcmp(current->id->str, "score") == 0 && !score_written) 
	score_written = current->value.intvalue;
      if(strcmp(current->id->str, "bit_score") == 0 && !bit_score_written)
	bit_score_written = current->value.realvalue;
      current = current->next;
    }
  if(score_written && bit_score_written)
    printf("Score = %.1f bits (%d),\n", bit_score_written, score_written);
  else
    printf("ERROR!!! didnt find score or score was zero\n");
}

int MyWriteSeqId(void *ptr, int *fb, Q_Headers *headers, FILE *outfp)
{
  DbtagPtr id;
  int	i, j, k;
  int   headerbegin = 0;
  
  if (ptr == NULL)
    printf("ERROR: Emptry Sequence ID Pointer\n");
  else {
    id = (DbtagPtr) ptr;
    for(i=1; i<fb[0]; i+=2) {
      for(j=fb[i]; j<=fb[i+1]; j++) {
	if(headerbegin == id->tag->id) {
	  if (last_seq != j) {
	    if(Length(headers, j) > 60)
	      printf("%60.60s...\n", Header(headers, j));
	    else
	      for(k=0; k<Length(headers,j); k++)
		printf("%c", *(Header(headers,j)+k));
	    last_seq = j;
	  }
	  return(headerbegin);
	}
	headerbegin++;
      }
    }
  }
  return(-1);
}


int MyWriteSeqIds(SeqIdPtr sip, int *fb, Q_Headers *headers, FILE *outfp)
{
  int seq_num = 0;
  if(sip == NULL)
    printf("ERROR: No Sequence ID pointer!\n");
  else
    while(sip != NULL) {
      seq_num = MyWriteSeqId(sip->data.ptrvalue, fb, headers, outfp);
      sip = sip->next;
    }
  return(seq_num);
}

void MyWriteAlign(int strand, int start, int len, char *seq, FILE *outfp)
{
  int	i;
  static int		convert[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
				       0, 2, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 3};
  static char		alphabet[] = "TGCA";

#ifdef DEBUG
  printf("last_seq: %d\n", last_seq);
  printf("strand: %d\n", strand);
  printf("start: %d \n", start);
  printf("len: %d\n", len);
  printf("seq: %s\n", seq);
#endif
  if(start == -1)
    printf("-");
  else {
    for(i=0; i<len; i++){
      if(strand == 1){
	if (seq[start+i] != '\0')
	  printf("%c", seq[start+i]);
	/* CHANGE!!! RANGE CHECKING??? PUT SOMETHING MORE ERROR PROOF HERE! */
      }
      if(strand == 2){
	printf("%c", alphabet[convert[(int)seq[start+len-i-1]]]);
	/* CHANGE!!! RANGE CHECKING??? PUT SOMETHING MORE ERROR PROOF HERE! */
      }
    }
  }
}


int MyWriteSegs(DenseSegPtr dsp, char *query, Q_SSet *db, int *fb, Q_Headers *headers, FILE *outfp)
  /*  typedef struct denseg {
      Int2	dim,
		numseg;
      SeqIdPtr	ids;           dimension is dim
      Int4Ptr starts;	       dimension is dim * numseg
      Int4Ptr lens;	       dimension is numseg 
      Uint1Ptr strands;	       dimension is dim * numseg
      ScorePtr scores;	       dimension is numseg
      } DenseSeg, PNTR DenseSegPtr;
  */
{
  int	i;
  int	seq_num = 0;

  if(dsp == NULL)
    printf("NO SEGS!!!!\n");
  else {
    /*  Unimportant Values
	fprintf(outfp, "dsp->dim:    %d\n", dsp->dim);
	fprintf(outfp, "dsp->numseg: %d\n", dsp->numseg);
	fprintf(outfp, "dsp->dim:    %d\n", dsp->dim);
    */
    seq_num = MyWriteSeqIds(dsp->ids, fb, headers, outfp);
    if(dsp->dim != 2)
      printf("ERROR: DIMENSION != 2 !!! \n");
    else {
      printf("Query: ");
      for(i=0; i<dsp->numseg; i++) {
	if(dsp->strands[2*i] == 2) {
	  dsp->strands[2*i] = 1;
	  if(dsp->strands[2*i+1] == 2)
	    printf("ERROR!!! double 2 strands\n");
	  else
	    dsp->strands[2*i+1] = 2;
	}
	MyWriteAlign(dsp->strands[2*i], dsp->starts[2*i], dsp->lens[i], query, outfp);
      }
      for(i=0; i<dsp->numseg; i++) {
	if(dsp->strands[2*i] == 1) {
	  printf("%7d ", dsp->starts[2*i]+1);
	  printf("- %7d, ", dsp->starts[2*i] + dsp->lens[i]);
	}
	else {
	  printf("%7d ", dsp->starts[2*i] + dsp->lens[i]);
	  printf("- %7d, ", dsp->starts[2*i]+1);
	}
      }
      printf("\nSbjct: ");
      for(i=0; i<dsp->numseg; i++) 
	MyWriteAlign(dsp->strands[2*i+1], dsp->starts[2*i+1], dsp->lens[i], Sequence(db, last_seq), outfp);
      for(i=0; i<dsp->numseg; i++) {
	if(dsp->strands[2*i+1] == 1) {
	  printf("%7d ", dsp->starts[2*i+1]+1);
	  printf("- %7d, ", dsp->starts[2*i+1] + dsp->lens[i]);
	}
	else {
	  printf("%7d ", dsp->starts[2*i+1] + dsp->lens[i]);
	  printf("- %7d, ", dsp->starts[2*i+1]+1);
	}
      }
      printf("\n");
    }
  }
  return seq_num;
}

void MyWriteHits(SeqAlignPtr seqalign, char *query, int*fb, Q_SSet *db, Q_Headers *hdrs, FILE *outfp)
  /*
    typedef struct seqalign {
    Uint1		type,
			segtype;
    Int2		dim;
    ScorePtr		score;
    Pointer		segs;
    struct seqalign PNTR next;
    SeqLocPtr		bounds;	   sequence of SeqLocPtr    
    SeqIdPtr		master;    for SAT_MASTERSLAVE
    SeqAlignIndexPtr	saip;      for added Alignment Indexing structures
    GatherIndex		idx;       internal gather/objmgr tracking fields
    Uint2		alignID;   unique number assigned to alignment 
    } SeqAlign, PNTR SeqAlignPtr;

   * SeqAlign.type values **
   #define SAT_GLOBAL 1		ordered segments, over full length of seqs 
   #define SAT_DIAGS 2		unordered, possibly overlapping segments 
   #define SAT_PARTIAL 3	ordered segments, over part of sequence 
   #define SAT_MASTERSLAVE 4    set of SeqAligns, all of which have one common 
				sequence. Not in ASN.1 yet 
   
   * SeqAlign.segtype values **
   #define SAS_DENDIAG 1
   #define SAS_DENSEG 2
   #define SAS_STD 3
   #define SAS_PACKED 4
   #define SAS_DISC 5
   #define SAS_COMPSEQ 6
   */
{
  int		i=0;
  SeqAlignPtr	current;

  last_seq = -1;		/* No sequence printed yet */
  current = seqalign;

  while(current != NULL) {
    /* Unimportant Values
       printf("type   : %10d\n", current->type);
       printf("segtype: %10d\n", current->segtype);
       printf("dim    : %10d\n", current->dim);
       MyWriteSeqLocPtr(current->bounds, outfp);
       MyWriteSeqIdPtr(current->master, outfp);
       MyWriteSeqAlignIndexPtr(current->saip, outfp);
       MyWriteGatherIndex(current->idx, outfp);
       printf("alignID: %10d\n", current->alignID);
    */
    MyWriteSegs((DenseSegPtr)current->segs, query, db, fb, hdrs, outfp);
    MyWriteScore(current->score, outfp);
    current = current->next;
    i++;
  }
}

void MyPrintOptions(BLAST_OptionsBlkPtr options)
{
  printf("%f\n",options->gap_decay_rate);
  printf("%f\n",options->gap_prob);
  printf("%d\n",options->gap_size);
  printf("%d\n",options->window_size);
  printf("%d\n",options->threshold_first);
  printf("%d\n",options->threshold_second);
  printf("%f\n",options->expect_value);
  printf("%f\n",options->e2);
  printf("%d\n",options->cutoff_s);
  printf("%d\n",options->cutoff_s2);
  printf("%d\n",options->two_pass_method);
  printf("%d\n",options->multiple_hits_only);
  printf("%d\n",options->hitlist_size);
  printf("%d\n",options->number_of_bits);
  printf("%d\n",options->dropoff_1st_pass);
  printf("%d\n",options->dropoff_2nd_pass);
  printf("%d\n",options->number_of_cpus);
  if(options->matrix != NULL)
    printf("%s\n",options->matrix);
  else
    printf("options->matrix = NULL!\n");
  printf("%d\n",options->old_stats);
  printf("%d\n",options->do_sum_stats);
  printf("%d\n",options->use_large_gaps);
  printf("%d\n",options->wordsize);
  printf("%d\n",options->penalty);
  printf("%d\n",options->reward);
  printf("%d\n",options->genetic_code);
  printf("%d\n",options->db_genetic_code);
  printf("%d\n",options->filter);
  if(options->filter_string != NULL)
    printf("%s\n",options->filter_string);
  else
    printf("options->filter_string = NULL!\n");
  printf("%d\n",options->gapped_calculation);
  printf("%d\n",options->gap_open);
  printf("%d\n",options->gap_extend);
  printf("%d\n",options->gap_x_dropoff);
  printf("%d\n",options->gap_x_dropoff_final);
  printf("%d\n",options->decline_align);
  printf("%f\n",options->gap_trigger);
  printf("%d\n",options->discontinuous);
  printf("%d\n",options->required_start);
  printf("%d\n",options->required_end);
  printf("%d\n",options->db_length);
  printf("%d\n",options->dbseq_num);
  printf("%d\n",options->searchsp_eff);
  printf("%d\n",options->ethresh);
  printf("%d\n",options->maxNumPasses);
  printf("%d\n",options->pseudoCountConst);
  printf("%s\n",options->program_name);
  printf("%d\n",options->cpu_limit);
  printf("%d\n",options->hsp_range_max);
  printf("%d\n",options->block_width);
  printf("%d\n",options->perform_culling);
  printf("%d\n",options->isPatternSearch);
  if(options->gifile != NULL)
    printf("%s\n",options->gifile);
  else
    printf("options->gifile = NULL!\n");
  if(options->gilist != NULL)
    printf("%s\n",options->gilist);
  else
    printf("options->gilist = NULL!\n");
  printf("%d\n",options->do_not_reevaluate);
  printf("%d\n",options->first_db_seq);
  printf("%d\n",options->final_db_seq);
  if(options->entrez_query != NULL)
    printf("%s\n",options->entrez_query);
  else
    printf("options->entrez_query = NULL!\n");
  if(options->org_name != NULL)
    printf("%s\n",options->org_name);
  else
    printf("options->org_name = NULL!\n");
  printf("%d\n",options->strand_option);
  printf("%d\n",options->hsp_num_max);
  printf("%d\n",options->tweak_parameters);
  printf("%d\n",options->smith_waterman);
  if(options->phi_pattern != NULL)
    printf("%s\n",options->phi_pattern);
  else
    printf("options->phi_pattern = NULL!\n");
  printf("%d\n",options->use_real_db_size);
  printf("%d\n",options->use_best_align);
  printf("%d\n",options->max_num_patterns);
  printf("%d\n",options->theCacheSize);
  printf("%d\n",options->no_check_score);
}


BLAST_OptionsBlkPtr MySetOptions()
{  
  BLAST_OptionsBlkPtr options;
  
  options = (BLAST_OptionsBlkPtr) MemNew(sizeof(BLAST_OptionsBlk));
  options->gap_decay_rate = 0.5;
  options->gap_prob = 0.5;
  options->gap_size = 50;
  options->window_size = 0;
  options->threshold_first = 0;
  options->threshold_second = 0;
  options->expect_value = 10.0;
  options->e2 = 0.05;
  options->cutoff_s = 0;
  options->cutoff_s2 = 0;
  options->two_pass_method = 0;
  options->multiple_hits_only = 0;
  options->hitlist_size = MAX_HITS;
  options->number_of_bits = 0;
  options->dropoff_1st_pass = 0;
  options->dropoff_2nd_pass = 20;
  options->number_of_cpus = 1;
  options->matrix = NULL;
  options->old_stats = 0;
  options->do_sum_stats = 1;
  options->use_large_gaps = 0;
  options->wordsize = 11;
  options->penalty = -3;
  options->reward = 1;
  options->genetic_code = 1;
  options->db_genetic_code = 1;
  options->filter = 0;
  options->filter_string = StringSave("D");
  options->gapped_calculation = 1;
  options->gap_open = 5;
  options->gap_extend = 2;
  options->gap_x_dropoff = 20;
  options->gap_x_dropoff_final = 50;
  options->decline_align = 32767;
  options->gap_trigger = 25.0;
  options->discontinuous = 0;
  options->required_start = 0;
  options->required_end = -1;
  options->db_length = 0;
  options->dbseq_num = 0;
  options->searchsp_eff = 0;
  options->ethresh = 0;
  options->maxNumPasses = 0;
  options->pseudoCountConst = 0;
  options->program_name = StringSave("blastn");;
  options->cpu_limit = 0;
  options->hsp_range_max = 100;
  options->block_width = 20;
  options->perform_culling = 1;
  options->isPatternSearch = 0;
  options->gifile = NULL;
  options->gilist = NULL;
  options->do_not_reevaluate = 0;
  options->first_db_seq = 0;
  options->final_db_seq = 0;
  options->entrez_query = NULL;
  options->org_name = NULL;
  options->strand_option = 3;
  options->hsp_num_max = 0;
  options->tweak_parameters = 0;
  options->smith_waterman = 0;
  options->phi_pattern = NULL;
  options->use_real_db_size = 0;
  options->use_best_align = 0;
  options->max_num_patterns = 0;
  options->theCacheSize = 0;
  options->no_check_score = 0;
  return options;
}


char **MySetupArgs()
{
  char	**argc;
  argc = (char **)MyMalloc(sizeof (char *) * 10, "artificial argc");
  argc[0] = argvec;
  argc[1] = argc[0] + 9;
  argc[2] = argc[1] + 3;
  argc[3] = argc[2] + 7;
  argc[4] = argc[3] + 3;
  argc[5] = argc[4] + 12;
  argc[6] = argc[5] + 3;
  argc[7] = argc[6] + 12;
  argc[8] = argc[7] + 3;
  argc[9] = NULL;
			   
  return argc;
}

SeqEntryPtr ReadQuery(char *query)
{
  SeqEntryPtr   sep = NULL;
  BioseqPtr     bsp = NULL;
  int		query_len;
  char		*qbuffer;
  int		qbuffer_length;
  register int	shift;
  register int  i;
  static int		convert[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
				       0, 2, 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 0, 0, 0, 3};

  query_len = strlen(query);
  printf("query has length %d\n", query_len);

  if((sep = SeqEntryNew()) == NULL)
    {
      printf("ERROR: out of virtual memory allocating query sequence\n");
      exit(-3);
    }
  sep->choice = 1;  /* == BIOSEQ */
  if((bsp = BioseqNew()) == NULL) 
    {
      printf("ERROR: out of virtual memory allocating query sequence\n");
      exit(-3);
    }
  /* Initializing Seq-entry structure */
  sep->data.ptrvalue = bsp;
  SeqMgrSeqEntry (SM_BIOSEQ, (Pointer)bsp, sep); 
  bsp->mol = Seq_mol_na;
  bsp->seq_data_type = Seq_code_ncbi4na;
  bsp->repr = Seq_repr_raw;
  SeqMgrAddToBioseqIndex (bsp);
  bsp->id = MakeNewProteinSeqId (NULL, NULL); /* create id for unknown seq */
  /* OK, now processing sequence */

  qbuffer_length = query_len/2 + query_len%2;
  qbuffer = (char *)MyMalloc(qbuffer_length, "query sequence");

  bsp->length = query_len; 
  bsp->seq_data = BSNew(query_len);
  for(i=0; i<qbuffer_length; i++)
    qbuffer[i] = 0;
  for(i=0; i<query_len; i++) {
    shift = 4 * ((i+1)%2);
    qbuffer[i>>1] += (convert[(int)query[i]] << shift);
  }

  BSWrite(bsp->seq_data, qbuffer, qbuffer_length);
  free(qbuffer);
 
  BioseqPack(bsp);     /* Trying to pack Bioseq more */
  return sep;
}

#define NUMARG 31

static Args myargs [NUMARG] = {
 { "Program Name",
        NULL, NULL, NULL, FALSE, 'p', ARG_STRING, 0.0, 0, NULL},
  { "Database", 
	"nr", NULL, NULL, FALSE, 'd', ARG_STRING, 0.0, 0, NULL},
  { "Query File", 
	"stdin", NULL, NULL, FALSE, 'i', ARG_FILE_IN, 0.0, 0, NULL},
  { "Expectation value (E)", 
	"10.0", NULL, NULL, FALSE, 'e', ARG_FLOAT, 0.0, 0, NULL},
  { "alignment view options:\n0 = pairwise,\n1 = master-slave showing identities,\n2 = master-slave no identities,\n3 = flat master-slave, show identities,\n4 = flat master-slave, no identities,\n5 = master-slave no identities and blunt ends,\n6 = flat master-slave, no identities and blunt ends", 
        "0", NULL, NULL, FALSE, 'm', ARG_INT, 0.0, 0, NULL},
  { "BLAST report Output File", 
	"stdout", NULL, NULL, TRUE, 'o', ARG_FILE_OUT, 0.0, 0, NULL},
  { "Filter query sequence (DUST with blastn, SEG with others)",
        "T", NULL, NULL, FALSE, 'F', ARG_STRING, 0.0, 0, NULL},
  { "Cost to open a gap (zero invokes default behavior)",
	"0", NULL, NULL, FALSE, 'G', ARG_INT, 0.0, 0, NULL},
  { "Cost to extend a gap (zero invokes default behavior)",
	"0", NULL, NULL, FALSE, 'E', ARG_INT, 0.0, 0, NULL},
  { "X dropoff value for gapped alignment (in bits) (zero invokes default behavior)",
	"0", NULL, NULL, FALSE, 'X', ARG_INT, 0.0, 0, NULL},
  { "Show GI's in deflines",
        "F", NULL, NULL, FALSE, 'I', ARG_BOOLEAN, 0.0, 0, NULL},
  { "Penalty for a nucleotide mismatch (blastn only)",
	"-3", NULL, NULL, FALSE, 'q', ARG_INT, 0.0, 0, NULL},
  { "Reward for a nucleotide match (blastn only)",
	"1", NULL, NULL, FALSE, 'r', ARG_INT, 0.0, 0, NULL},
  { "Number of one-line descriptions (V)",
        "500", NULL, NULL, FALSE, 'v', ARG_INT, 0.0, 0, NULL},
  { "Number of alignments to show (B)",
        "250", NULL, NULL, FALSE, 'b', ARG_INT, 0.0, 0, NULL},
  { "Threshold for extending hits, default if zero",
        "0", NULL, NULL, FALSE, 'f', ARG_INT, 0.0, 0, NULL},
  { "Perfom gapped alignment (not available with tblastx)",
        "T", NULL, NULL, FALSE, 'g', ARG_BOOLEAN, 0.0, 0, NULL},
  { "Query Genetic code to use",
        "1", NULL, NULL, FALSE, 'Q', ARG_INT, 0.0, 0, NULL},
  { "DB Genetic code (for tblast[nx] only)",
        "1", NULL, NULL, FALSE, 'D', ARG_INT, 0.0, 0, NULL},
  { "Number of processors to use",
        "1", NULL, NULL, FALSE, 'a', ARG_INT, 0.0, 0, NULL},
  { "SeqAlign file", 
	NULL, NULL, NULL, TRUE, 'O', ARG_FILE_OUT, 0.0, 0, NULL},
  { "Believe the query defline",
        "F", NULL, NULL, FALSE, 'J', ARG_BOOLEAN, 0.0, 0, NULL},
  { "Matrix",
        "BLOSUM62", NULL, NULL, FALSE, 'M', ARG_STRING, 0.0, 0, NULL},
  { "Word size, default if zero", 
        "0", NULL, NULL, FALSE, 'W', ARG_INT, 0.0, 0, NULL},
  { "Effective length of the database (use zero for the real size)", 
        "0", NULL, NULL, FALSE, 'z', ARG_FLOAT, 0.0, 0, NULL},
  { "Number of best hits from a region to keep",
        "100", NULL, NULL, FALSE, 'K', ARG_INT, 0.0, 0, NULL},
  { "Length of region used to judge hits",
        "20", NULL, NULL, FALSE, 'L', ARG_INT, 0.0, 0, NULL},
  { "Effective length of the search space (use zero for the real size)",
        "0", NULL, NULL, FALSE, 'Y', ARG_FLOAT, 0.0, 0, NULL},
  { "Query strands to search against database (for blast[nx], and tblastx).  3 is both, 1 is top, 2 is bottom",
        "3", NULL, NULL, FALSE, 'S', ARG_INT, 0.0, 0, NULL},
  { "Produce HTML output",
        "F", NULL, NULL, FALSE, 'T', ARG_BOOLEAN, 0.0, 0, NULL},
  { "Restrict search of database to list of GI's",
	NULL, NULL, NULL, TRUE, 'l', ARG_STRING, 0.0, 0, NULL}
};

static ReadDBFILEPtr
Myreaddb_new_internal(CharPtr filename, Uint1 is_prot, Uint1 init_state, CommonIndexHeadPtr cih)
/* 
	filename: name of the file to be openend.
	is_prot: three choices: protein, nucleotide, or either one.
	init_state: how much should be initialized.
		READDB_NEW_DO_ALL : initialize everything possible
		READDB_NEW_DO_REPORT : init enough for a report on db size etc.
	cih: common index
*/
{
  ReadDBFILEPtr rdfp=NULL;
  Char buffer[PATH_MAX];
  Char	database_dir[PATH_MAX] = "";
  Uint4 seq_type, formatdb_ver, date_length, title_length, value;
  Int4 num_seqs;
  CharPtr	charptr;

  rdfp = ReadDBFILENew();
  if (rdfp == NULL)
    return NULL;
    
  rdfp->filename = StringSave("FILTERED_DB");
  rdfp->is_prot = is_prot;
  sprintf(rdfp->full_filename, "FILTERED_DB");
  
  /* construct full file name */
  /*
  if (!StringCmp(database_dir, "")) {
      sprintf(rdfp->full_filename, "%s", Nlm_FileNameFind(filename));
  }
  else if (!TRUE) {
      sprintf(rdfp->full_filename, "%s%s%s", database_dir, DIRDELIMSTR, filename);
  } else  {
      sprintf(rdfp->full_filename, "%s%s%s", database_dir, DIRDELIMSTR, Nlm_FileNameFind(filename));
  }
  */
  /* check if present main three files: index, sequences, headers */
  
  sprintf(buffer, "%s.%cin", rdfp->full_filename, rdfp->is_prot? 'p':'n');
  if((rdfp->indexfp = MyNlmOpenMFILE(memdb[0])) == NULL) {
    ErrPostEx(SEV_WARNING, 0, 0, "Unable to open %s", buffer);
    rdfp = readdb_destruct(rdfp);
    return rdfp;
  }
  
  sprintf(buffer, "%s.%csq", rdfp->full_filename, rdfp->is_prot? 'p':'n');
  if((rdfp->sequencefp = MyNlmOpenMFILE(memdb[2])) == NULL) {
    ErrPostEx(SEV_WARNING, 0, 0, "Unable to open %s", buffer);
    rdfp = readdb_destruct(rdfp);
    return FALSE;
  } 
  sprintf(buffer, "%s.%chr", rdfp->full_filename, rdfp->is_prot? 'p':'n');
  if((rdfp->headerfp = MyNlmOpenMFILE(memdb[1])) == NULL) {
    ErrPostEx(SEV_WARNING, 0, 0, "Unable to open %s", buffer);
    rdfp = readdb_destruct(rdfp);
    return FALSE;
  } 

  /*
  printf("rdfp->indexfp->mmp: %d / db[0]: %d\n", rdfp->indexfp->mmp, memdb[0]);
  printf("rdfp->sequencefp->mmp: %d / db[2]: %d\n", rdfp->sequencefp->mmp, memdb[2]);
  printf("rdfp->headerfp->mmp: %d / db[1]: %d\n", rdfp->headerfp->mmp, memdb[1]);
  */

  /* fill in other fields of rdfp-> */

  NlmReadMFILE((Uint1Ptr) &value, 4, 1, rdfp->indexfp);
  formatdb_ver = Nlm_SwapUint4(value);
  if (formatdb_ver != FORMATDB_VER) {
    ErrPostEx(SEV_WARNING, 0, 0, "readdb: wrong version of formatdb "
              "was used to make database.");
    rdfp = readdb_destruct(rdfp);
    return NULL;
  }
  rdfp->formatdb_ver = FORMATDB_VER;
  
  NlmReadMFILE((Uint1Ptr) &value, 4, 1, rdfp->indexfp);
  seq_type = Nlm_SwapUint4(value);
  /* printf("seq_type: %d\n", seq_type);*/
  if ((rdfp->is_prot && seq_type == 0) || (!rdfp->is_prot && seq_type == 1)) {
    rdfp = readdb_destruct(rdfp);
    return rdfp;
  }
  NlmReadMFILE((Uint1Ptr) &value, 4, 1, rdfp->indexfp);
  title_length = Nlm_SwapUint4(value);
  /* printf("Title length: %d\n", title_length); */
  rdfp->title = StringSave("RAM_DB");;

  
  NlmReadMFILE((Uint1Ptr) &value, 4, 1, rdfp->indexfp);
  date_length = Nlm_SwapUint4(value);

  /* printf("Date Length: %d\n", date_length); */
  
  rdfp->date = (CharPtr)Nlm_Malloc((date_length+1)*sizeof(Char));
  NlmReadMFILE((Uint1Ptr) rdfp->date, date_length, 1, rdfp->indexfp);
  rdfp->date[date_length] = NULLB;
  /*
  printf("Date: %s\n", rdfp->date);
  printf("rdfp->indexfp->mmp %d\n", rdfp->indexfp->mmp);
  */
  NlmReadMFILE((Uint1Ptr) &(value), 4, 1, rdfp->indexfp);
  num_seqs = rdfp->num_seqs = Nlm_SwapUint4(value);
  /* printf("num_seqs: %d\n", num_seqs); */
  NlmReadMFILE((Uint1Ptr) &(value), 4, 1, rdfp->indexfp);
  rdfp->totlen = Nlm_SwapUint4(value);
  /* printf("rdfp->totlen: %d\n", rdfp->totlen); */
  NlmReadMFILE((Uint1Ptr) &(value), 4, 1, rdfp->indexfp);
  rdfp->maxlen = Nlm_SwapUint4(value);
  /* printf("rdfp->maxlen: %d\n", rdfp->maxlen); */

  if (!((title_length + date_length)%4) && rdfp->indexfp->mfile_true) {
      rdfp->header_index = (Uint4Ptr) rdfp->indexfp->mmp;
      rdfp->indexfp->mmp += 4 * (num_seqs+1);

      rdfp->sequence_index = (Uint4Ptr) rdfp->indexfp->mmp;
      rdfp->indexfp->mmp += 4 * (num_seqs+1);

      rdfp->ambchar_index = (Uint4Ptr) rdfp->indexfp->mmp;
      rdfp->indexfp->mmp += 4 * (num_seqs+1);
  } else {
      /* Use old stuff */
   
      if((rdfp->header_index = 
          (Uint4Ptr) Nlm_Malloc((num_seqs+1)*sizeof(Uint4))) == NULL) {
          rdfp = readdb_destruct(rdfp);
          return rdfp;
      }
      
      rdfp->header_index_start = rdfp->header_index;
      rdfp->header_index_offset = NlmTellMFILE(rdfp->indexfp);
      NlmReadMFILE((Uint1Ptr) rdfp->header_index, 4, num_seqs+1, rdfp->indexfp);
      
      if((rdfp->sequence_index = 
          (Uint4Ptr)Nlm_Malloc((num_seqs+1)*sizeof(Uint4))) == NULL) {
          rdfp = readdb_destruct(rdfp);
          return rdfp;
      }
      rdfp->sequence_index_start = rdfp->sequence_index;
      NlmReadMFILE((Uint1Ptr) rdfp->sequence_index, 4, num_seqs+1, rdfp->indexfp);
      
      if((rdfp->ambchar_index = (Uint4Ptr)Nlm_Malloc((num_seqs+1)*sizeof(Uint4))) == NULL) {
	rdfp = readdb_destruct(rdfp);
	return rdfp;
      }
      rdfp->ambchar_index_start = rdfp->ambchar_index;
      NlmReadMFILE((Uint1Ptr) rdfp->ambchar_index, 4, num_seqs+1, rdfp->indexfp);
  }

  /* Contents were allocated above. */
  rdfp->contents_allocated = TRUE;
  
  /* Initialize shared information structure */
  rdfp->shared_info = (ReadDBSharedInfoPtr) Nlm_Malloc(sizeof(ReadDBSharedInfo));
  rdfp->shared_info->nthreads = 0;
  rdfp->shared_info->headerfp = NULL;
  rdfp->shared_info->sequencefp = NULL;

  return rdfp;
}

/*
	Use by HeapSort (in BioseqBlastEngine) to rank Hitlist's.
*/

static int LIBCALLBACK
evalue_compare_hits(VoidPtr v1, VoidPtr v2)

{
    BLASTResultHitlistPtr h1, h2;
    BLASTResultHitlistPtr *hp1, *hp2;
    
    hp1 = (BLASTResultHitlistPtr *) v1;
    hp2 = (BLASTResultHitlistPtr *) v2;
    h1 = *hp1;
    h2 = *hp2;
    
    /* Sort first by evalue, then by score in case all evalues are zero. */

    if (h1->best_evalue < h2->best_evalue)
        return -1;
    if (h1->best_evalue > h2->best_evalue)
        return 1;
    if (h1->high_score > h2->high_score)
        return -1;
    if (h1->high_score < h2->high_score)
        return 1;
    
    /* In case of equal scores and E-values order will be determined by
       subject id */
    
    if (h1->subject_id > h2->subject_id)
        return -1;
    if (h1->subject_id < h2->subject_id)
        return 1;
    
    return 0;
}

#define AWAKE_THR_MIN_SIZE 6000000000.0 
SeqAlignPtr LIBCALL
MyBioseqBlastEngineCore(BlastSearchBlkPtr search, BLAST_OptionsBlkPtr options, Int4Ptr *pos_matrix)
{
	BLASTResultHspPtr hsp;
	BLASTResultsStructPtr result_struct;
	BLASTResultHitlistPtr   result_hitlist;
	GapXEditBlockPtr edit_block;
	Int4 hitlist_count, hitlist_max, hspcnt, index, index1, length, sequence_length;
	Int4 my_sequence_length;
	SeqAlignPtr sap, head, seqalign, seqalign_var;
	SeqIdPtr gi_list=NULL, subject_id;
	Uint1Ptr sequence, my_sequence, my_sequence_start;
	StdSegPtr ssp;
        
	head = seqalign = NULL;

start_timer;
	if (search == NULL || search->query_invalid)
		return NULL;


	/* If pos_matrix is not NULL, then psi-blast iterations are being 
	performed.  The first psi-blast iteration should be with normal
	blast. */
	if (pos_matrix)
	{
		search->sbp->posMatrix = pos_matrix;
		search->positionBased = TRUE;
                search->sbp->kbp = search->sbp->kbp_psi;
                search->sbp->kbp_gap = search->sbp->kbp_gap_psi;
		hitlist_max = search->result_struct->hitlist_max;
                search->result_struct = BLASTResultsStructDelete(search->result_struct);
		search->result_struct = BLASTResultsStructNew(hitlist_max, search->pbp->max_pieces, search->pbp->hsp_range_max);
                if (search->allocated & BLAST_SEARCH_ALLOC_WFP_FIRST)
		{
                       search->wfp_first = BLAST_WordFinderDestruct(search->wfp_first);
		       search->wfp_first = BLAST_WordFinderNew(search->sbp->alphabet_size,options->wordsize,1, FALSE, search->pbp->theCacheSize);
		}

		if (search->allocated & BLAST_SEARCH_ALLOC_WFP_SECOND)
		{
		       search->wfp_second = BLAST_WordFinderDestruct(search->wfp_second);
		       search->wfp_second = BLAST_WordFinderNew(search->sbp->alphabet_size,options->wordsize,1, FALSE, search->pbp->theCacheSize);
		}

		/* Only find words once if thresholds are the same. */
                 search->wfp = search->wfp_first;
		 if (search->whole_query == TRUE)
                 	BlastNewFindWords(search, 0, search->context[search->first_context].query->length, search->pbp->threshold_first, (Uint1) 0);
		 else
                 	BlastNewFindWords(search, search->required_start, search->required_end, search->pbp->threshold_first, (Uint1) 0);
		 lookup_position_aux_destruct(search->wfp->lookup);
		 if (search->pbp->threshold_first != search->pbp->threshold_second)
		 {
                 	search->wfp = search->wfp_second;
			if (search->whole_query == TRUE)
                    		BlastNewFindWords(search, 0, search->context[search->first_context].query->length, search->pbp->threshold_second, (Uint1) 0);

			else
                    		BlastNewFindWords(search, search->required_start, search->required_end, search->pbp->threshold_second, (Uint1) 0);
			lookup_position_aux_destruct(search->wfp->lookup);
		 }
		 else
		 {
			search->wfp_second = search->wfp_first;
		 }
	}

	/* Starting awake thread if multithreaded. */
	if (search->searchsp_eff > AWAKE_THR_MIN_SIZE)
		BlastStartAwakeThread(search->thr_info);

	stop_timer("BioseqBlastEngineCore: before do_the_blast_run()");

	printf("Before the blast run\n");
	do_the_blast_run(search);
	printf("After the blast run\n");
start_timer;
	head = NULL;
	if (StringCmp(search->prog_name, "blastn") == 0 && 
		search->pbp->gapped_calculation)
        {
		search->sbp->kbp_gap[search->first_context] = search->sbp->kbp[search->first_context];

		search->pbp->gap_open = options->gap_open;
		search->pbp->gap_extend = options->gap_extend;
/*
		search->pbp->gap_x_dropoff = (BLAST_Score) (options->gap_x_dropoff*NCBIMATH_LN2 / search->sbp->kbp_gap[search->first_context]->Lambda);
		search->pbp->gap_x_dropoff_final = (BLAST_Score) (options->gap_x_dropoff_final*NCBIMATH_LN2 / search->sbp->kbp_gap[search->first_context]->Lambda);
*/


		result_struct = search->result_struct;
       		hitlist_count = result_struct->hitlist_count;

		sequence=NULL;
		sequence_length=0;
		my_sequence=NULL;
		my_sequence_length=0;
		my_sequence_start=NULL;
		for (index=0; index<hitlist_count; index++)
		{
			length = readdb_get_sequence_ex(search->rdfp, result_struct->results[index]->subject_id, &sequence, &sequence_length);
			if (sequence_length > my_sequence_length-1)
			{
				my_sequence_start = MemFree(my_sequence_start);

				my_sequence_length = sequence_length+1;
				my_sequence_start = MemNew((my_sequence_length)*sizeof(Uint1));
				my_sequence_start[0] = ncbi4na_to_blastna[0];
				my_sequence = my_sequence_start+1;
			}
			for (index1=0; index1<length; index1++)
			{
				my_sequence[index1] = ncbi4na_to_blastna[sequence[index1]];
			}
			/* Gap character in last space. */
			my_sequence[length] = ncbi4na_to_blastna[0];
			seqalign = SumBlastGetGappedAlignmentTraceback(search, index, FALSE, FALSE, my_sequence, length);
			result_struct->results[index]->seqalign = seqalign;
		}
		sequence = MemFree(sequence);
		my_sequence_start = MemFree(my_sequence_start);
		search->sbp->kbp_gap[search->first_context] = NULL;

		HeapSort(result_struct->results, hitlist_count, sizeof(BLASTResultHitlistPtr), evalue_compare_hits);

		/* 
		The next loop organizes the SeqAligns (and the alignments in the
		BLAST report) in the same order as the deflines.
		*/
		head = NULL;
		for (index=0; index<hitlist_count; index++)
		{
		    seqalign = result_struct->results[index]->seqalign;
		    if (seqalign)
		    {
			if (head == NULL)
			{
				head = seqalign;
			}
			else
			{
				for (seqalign_var=head; seqalign_var->next;)
               		                seqalign_var = seqalign_var->next;
               		        seqalign_var->next = seqalign;
               	        }
		    }
		}
	}
	else if (search->pbp->gapped_calculation)
	{
		result_struct = search->result_struct;
                hitlist_count = result_struct->hitlist_count;
                for (index=0; index<hitlist_count; index++)
                {
                     seqalign = BlastGetGapAlgnTbckWithReaddb(search, index, FALSE);
		     result_struct->results[index]->seqalign = seqalign;
                }

		HeapSort(result_struct->results, hitlist_count, sizeof(BLASTResultHitlistPtr), evalue_compare_hits);

		/* 
		The next loop organizes the SeqAligns (and the alignments in the
		BLAST report) in the same order as the deflines.
		*/
		head = NULL;
		for (index=0; index<hitlist_count; index++)
		{
		    seqalign = result_struct->results[index]->seqalign;
		    if (seqalign)
		    {
			if (head == NULL)
			{
				head = seqalign;
			}
			else
			{
				for (seqalign_var=head; seqalign_var->next;)
               		                seqalign_var = seqalign_var->next;
               		        seqalign_var->next = seqalign;
               	        }
		    }
		}
	}
	else
	{
	  head = GetSeqAlignForResultHitList(search, TRUE, FALSE, options->discontinuous, FALSE, FALSE);
	}
	gi_list = SeqIdSetFree(gi_list);

	/* Stop the awake thread. */
	BlastStopAwakeThread(search->thr_info);

stop_timer("BioseqBlastEngineCore: after do_the_blast_run()");
	return head;
}

int CallBlast(char *query, int *fb, Q_SSet *db, Q_Headers *hdrs, int **blastdb, FILE *outfp)
{
  Boolean done = FALSE;
  Int4 start=0, stop=0;
  ReadDBFILEPtr new, tmp, var, rdfp_w_oidlist;
  CommonIndexHeadPtr	cih = NULL;
  
  Char lbuffer[PATH_MAX];

  Boolean multiple_hits;
  Int2 status, first_context, last_context;
  Int8	dblen;
  Int4	query_length;
  Int4		i;
  Nlm_FloatHi	searchsp_eff=0;
  Int4		virtual_mask_index;
  Uint4		virtual_oid_bit;
  ReadDBFILEPtr	tmprdfp;
  ReadDBFILEPtr	rdfp = NULL;
  OIDListPtr		virtual_oidlist = NULL;
  Int4		final_virtual_db_seq=0, final_db_seq=0;
  Int4		mask, oid, virtual_oid, maskindex, total_virtual_mask, base;
  BLASTContextStructPtr context;
  Uint1 is_prot;
  Int2 index;
  Uint1 alphabet;
  Int4 longest_db_seq=INT4_MAX;
  ReadDBFILEPtr rdfp_var;
  BlastSearchBlkPtr search;
  SeqLocPtr slp;

  BioseqPtr fake_bsp, query_bsp;
  BioSourcePtr source;
  BLAST_OptionsBlkPtr options;
  Boolean query_is_na;
  CharPtr PNTR last_char;
  SeqAlignPtr  seqalign;
  SeqEntryPtr sep;
  Uint4 align_options;
  ValNodePtr  other_returns, error_returns;
  
  CharPtr blast_program, blast_database, blast_inputfile;
  FILE *infp;
  
  memdb = blastdb;	/* MAKE blastdb GLOBAL */

  if (!setup_done) { /* THE CODE IN !setup_done SHOULD ONLY BE CALLED ONCE */
    /* Simulate an argument vector, own code */
    argv = MySetupArgs();
    setup_done = 1;
    /* THIS IS THE START OF ncbimain.c */
    Nlm_SetupArguments(argc, argv);
#ifdef MSC_VIRT
    if ( !_vheapinit(0, 1250, _VM_ALLSWAP) ) {
      ErrPost(CTX_NCBIOBJ, 1, "Can't open virtual memory");
      return 1;
    }
#endif
    /* THIS IS THE END OF ncbimain.c */
    if (! GetArgs ("blastall", NUMARG, myargs)) {
      return (1);
    }
    UseLocalAsnloadDataAndErrMsg ();
    if (! SeqEntryLoad())
      return 1;
    ErrSetMessageLevel(SEV_WARNING);
  }

  blast_program = myargs [0].strvalue;
  blast_database = myargs [1].strvalue;
  blast_inputfile = myargs [2].strvalue;
  
  if ((infp = FileOpen(blast_inputfile, "r")) == NULL)
    {
      ErrPostEx(SEV_FATAL, 0, 0, "blast: Unable to open input file %s\n", blast_inputfile);
      return (1);
    }
  
  query_is_na = TRUE;

  options = BLASTOptionNew(blast_program, (Boolean) myargs [16].intvalue);
  if (options == NULL)
    return 3;
  
  BLASTOptionSetGapParams(options, myargs[22].strvalue, 0, 0); 
  options->expect_value  = (Nlm_FloatHi) myargs [3].floatvalue;
  options->hitlist_size = MAX_HITS;
  if (myargs[7].intvalue != 0)
    options->gap_open = myargs[7].intvalue;
  if (myargs[8].intvalue != 0)
    options->gap_extend = myargs[8].intvalue;
  if (myargs[9].intvalue != 0)
    options->gap_x_dropoff = myargs[9].intvalue;
  options->filter_string = StringSave("D");
  options->penalty = myargs[11].intvalue;
  options->reward = myargs[12].intvalue;
  options->genetic_code = myargs[17].intvalue;
  options->db_genetic_code = myargs[18].intvalue;
  options->number_of_cpus = myargs[19].intvalue;
  if (myargs[23].intvalue != 0)
    options->wordsize = myargs[23].intvalue;
  options->hsp_range_max  = myargs[25].intvalue;
  if (options->hsp_range_max != 0)
    options->perform_culling = TRUE;
  options->block_width  = myargs[26].intvalue;
  if (myargs[27].floatvalue)
    options->searchsp_eff = (Nlm_FloatHi) myargs[27].floatvalue;
  options->strand_option = myargs[28].intvalue;


  options->db_length = db->offsets[db->size];

  align_options = 0;
  align_options += TXALIGN_COMPRESS;
  align_options += TXALIGN_END_NUM;
  align_options += TXALIGN_MATRIX_VAL;
  align_options += TXALIGN_SHOW_QS;
  
  /*
    sep = FastaToSeqBuffEx(query, last_char, TRUE, NULL, FALSE);
    if(sep != NULL)
  */
  while ((sep=FastaToSeqEntryEx(infp, query_is_na, NULL, FALSE)) != NULL) 
    {
      query_bsp = NULL;
      SeqEntryExplore(sep, &query_bsp, FindNuc);
      
      if (query_bsp == NULL) {
	ErrPostEx(SEV_FATAL, 0, 0, "Unable to obtain bioseq\n");
	return 2;
      }
      
      fake_bsp = BlastMakeFakeBioseq(query_bsp, NULL);
      
      source = BioSourceNew();
      source->org = OrgRefNew();
      source->org->orgname = OrgNameNew();
      source->org->orgname->gcode = options->genetic_code;
      ValNodeAddPointer(&(query_bsp->descr), Seq_descr_source, source);
      
      other_returns = NULL;
      error_returns = NULL;
      
#ifdef ORIGINAL_BIOSEQBLASTENGINE
      seqalign = BioseqBlastEngine(fake_bsp, blast_program, blast_database, options, &other_returns, &error_returns, tick_callback);
#endif
      /* THIS IS THE ALTERNATE BioseqBlastEngine CALL */
      slp = NULL;
      ValNodeAddPointer(&slp, SEQLOC_WHOLE, SeqIdDup(SeqIdFindBest(fake_bsp->id, SEQID_GI)));

#ifdef ORIGINAL_BLASTSETUPSEARCH
      search = BLASTSetUpSearchWithReadDbInternal (slp, NULL, blast_program, SeqLocLen(slp), blast_database, options, NULL, NULL, NULL, 0, NULL);
#endif
      /* THIS IS THE ALTERNATE BLASTSetUpSearchWithReadDbInternal CALL */
      if (options->window_size != 0)
	multiple_hits = TRUE;
      else
	multiple_hits = FALSE;

      BlastGetFirstAndLastContext(blast_program, slp, &first_context, &last_context, options->strand_option);

	if (slp)
		query_length = SeqLocLen(slp);
	else
		query_length = query_bsp->length;
		
	/* On the first call query length is used for the subject length. */
#ifdef ORIGINAL_BLASTSEARCHBLKNEWEXTRA
	search = BlastSearchBlkNewExtra(options->wordsize, query_length, blast_database, multiple_hits, options->threshold_first, options->threshold_second, options->hitlist_size, blast_program, NULL, first_context, last_context, rdfp, options->window_size, options->theCacheSize);
#endif
	/* THIS IS THE ALTERNATE BlastSearchBlkNewExtra CALL */
	search = (BlastSearchBlkPtr) MemNew(sizeof(BlastSearchBlk));
	if (search != NULL) {
	  search->allocated = 0;	/* everything's allocated here. */
	  search->allocated += BLAST_SEARCH_ALLOC_QUERY;
	  search->allocated += BLAST_SEARCH_ALLOC_SUBJECT;
	  search->allocated += BLAST_SEARCH_ALLOC_PBP;
	  search->allocated += BLAST_SEARCH_ALLOC_SBP;
	  search->allocated += BLAST_SEARCH_ALLOC_EWPPARAMS;
	  search->allocated += BLAST_SEARCH_ALLOC_CONTEXT;
	  search->allocated += BLAST_SEARCH_ALLOC_RESULTS;
	  search->allocated += BLAST_SEARCH_ALLOC_READDB;
	  search->allocated += BLAST_SEARCH_ALLOC_ALL_WORDS;
	  search->allocated += BLAST_SEARCH_ALLOC_THRINFO;
	  search->positionBased = FALSE;
	  alphabet = BLASTNA_SEQ_CODE;
	    
	  if (blast_database != NULL) {
	    if (rdfp == NULL) {
	      is_prot = READDB_DB_IS_NUC;
#ifdef READDB_FROM_FILE	      
	      search->rdfp=readdb_new_ex2(blast_database, is_prot, READDB_NEW_INDEX, NULL);
#endif
	      /* THIS IS THE ALTERNATE readdb_new_ex2 CALL */
	      new = NULL;
	      rdfp_w_oidlist = NULL;

	      while (!done)
		{
		  done = readdb_parse_db_names(&blast_database, lbuffer);
		  if (*lbuffer == NULLB)
		    break;

		  if(!(tmp = Myreaddb_new_internal(lbuffer, is_prot, READDB_NEW_INDEX, cih)))
		    
		    continue;
		  
		  if (tmp->cih) {
		    cih = tmp->cih;
		  }
		  
		  if (tmp->oidlist)
		    {
		      /* Save these separately. */
		      if (rdfp_w_oidlist == NULL)
			{
			  rdfp_w_oidlist = tmp;
			}
		      else
			{
			  var = rdfp_w_oidlist;
			  while (var->next)
			    var = var->next;
			  var->next = tmp;
			}
		      continue; /* do not execute rest of while loop. */
		    }
		  else if (tmp->not_first_time == FALSE)
		    {
		      tmp->not_first_time = TRUE;
		      
		      stop = tmp->num_seqs-1+start;
		      tmp->start = start;
		      tmp->stop = stop;
		      tmp->ambchar_index -= start;
		      tmp->header_index -= start;
		      tmp->sequence_index -= start;
		      
		      start += tmp->num_seqs;
		      
		      var = new;
		      if (new == NULL)
			{
			  new = tmp;
			}
		      else
			{
			  var = new;
			  while(var->next)
			    var = var->next;
			  var->next = tmp;
			}
		    }
		}
	      
	      /* adjust all the RDFP's with ordinal ID list. */
	      tmp = rdfp_w_oidlist;
	      while (tmp)
		{
		  stop = tmp->num_seqs-1+start;
		  tmp->start = start;
		  tmp->stop = stop;
		  tmp->ambchar_index -= start;
		  tmp->header_index -= start;
		  tmp->sequence_index -= start;
		  
		  start += tmp->num_seqs;
		  tmp = tmp->next;
		}
	      
	      /* Attach the RDFP's with an OID. */
	      if (rdfp_w_oidlist)
		{
		  if (new == NULL)
		    new = rdfp_w_oidlist;
		  else
		    {
		      var = new;
		      while(var->next)
			var = var->next;
		      var->next = rdfp_w_oidlist;
		    }
		}
	      
	      if (new)
		new->not_first_time = FALSE;
	      search->rdfp = new;
	      /* END OF THE ALTERNATE readdb_new_ex2 CALL */
	      if (search->rdfp == NULL) {
		printf("ERROR: Database pointer is NULL!\n");
		return 1;
	      }
	    }
	    else { /* Attaches to the rdfp, rather than reallocating it. */
	      search->rdfp = readdb_attach(rdfp);
	    }
	    
	    rdfp_var = search->rdfp;
	    longest_db_seq = 0;
	    while (rdfp_var)
	      {
		longest_db_seq = MAX(longest_db_seq, readdb_get_maxlen(rdfp_var));
		rdfp_var = rdfp_var->next;
	      }
	  }
	  
	  search->first_context = first_context;
	  search->last_context = last_context;
	  
	  search->pbp = 
	    (BLAST_ParameterBlkPtr) MemNew(sizeof(BLAST_ParameterBlk));
	  
	  search->pbp->theCacheSize = options->theCacheSize;
	  
	  search->sbp = BLAST_ScoreBlkNew(alphabet, last_context+1);
	  
	  if (multiple_hits)
	    search->wfp_second = BLAST_WordFinderNew(256, options->wordsize, READDB_COMPRESSION_RATIO, FALSE, search->pbp->theCacheSize);
	  else
	    search->wfp_second = BLAST_WordFinderNew(256, options->wordsize, READDB_COMPRESSION_RATIO, TRUE, search->pbp->theCacheSize);
	  search->allocated += BLAST_SEARCH_ALLOC_WFP_SECOND;
	  
	  search->prog_name = StringSave(blast_program);
	  search->prog_number = BlastGetProgramNumber(blast_program);
	  
	  search->ewp_params = BLAST_ExtendWordParamsNew(query_length, multiple_hits, options->window_size);
	  
	  context = search->context = (BLASTContextStructPtr) MemNew((1+search->last_context)*sizeof(BLASTContextStruct));
	  for (index=search->first_context; index<=search->last_context; index++)
	    {
	      context[index].ewp = BLAST_ExtendWordNew(search->ewp_params);
	      context[index].query = (BlastSequenceBlkPtr) MemNew(sizeof(BlastSequenceBlk));
	      context[index].query->frame = ContextToFrame(search, index);
	      context[index].query_allocated = TRUE;
	    }
	  
	  search->subject = (BlastSequenceBlkPtr) MemNew(sizeof(BlastSequenceBlk));
	  /* 100 is the size limit in the present BLAST for hsp's. */
	  search->hsp_array_size = 100;
	  /* The results are held here. */
	  search->result_size = options->hitlist_size;
	  /*
	    search->result_struct = BLASTResultsStructNew(options->hitlist_size, search->pbp->max_pieces, search->pbp->hsp_range_max);
	  */
	  
	  search->worst_evalue = DBL_MAX;
	  
	  search->whole_query = TRUE;
	  search->required_start = 0;
	  search->required_end = -1;
	  
	  search->all_words = NULL;
	  
	  search->thr_info = BlastThrInfoNew();
	  
#ifdef BLAST_COLLECT_STATS
	  search->first_pass_hits = 0;
	  search->second_pass_hits = 0;
	  search->second_pass_trys = 0;
	  search->first_pass_extends = 0;
	  search->second_pass_extends = 0;
	  search->first_pass_good_extends = 0;
	  search->second_pass_good_extends = 0;
	  search->number_of_seqs_better_E = 0;
	  search->prelim_gap_no_contest = 0;
	  search->prelim_gap_passed = 0;
	  search->prelim_gap_attempts = 0;
	  search->real_gap_number_of_hsps = 0;
#endif
	}

	/* END OF ALTERNATE BlastSearchBlkNewExtra CALL */
	if (search)
	  {
	    readdb_get_totals(search->rdfp, &(dblen), &(search->dbseq_num));
	    
	    /* Ok, we do not have a gi-list specified, but maybe
	       we have an a mask database in the list of databases,
	       we need to create one mask for all such databases */
	    
	    tmprdfp = search->rdfp;
	    while (tmprdfp) {
	      
	      final_virtual_db_seq = tmprdfp->stop;
	      if (!tmprdfp->oidlist)
		final_db_seq = tmprdfp->stop;
	      tmprdfp = tmprdfp->next;
	    }
	    
	    tmprdfp = search->rdfp;
	    while (tmprdfp) {
	      if (tmprdfp->oidlist) {
		if (!virtual_oidlist) {
				/* create new oidlist for virtual database */
		  virtual_oidlist = (OIDListPtr) MemNew(sizeof(OIDList));
		  virtual_oidlist->total = final_virtual_db_seq + 1;
		  total_virtual_mask = final_virtual_db_seq/MASK_WORD_SIZE + 2;
		  virtual_oidlist->list = (Uint4Ptr) MemNew (total_virtual_mask*sizeof(Int4));
		}
		/* Now populate the virtual_oidlist */
		maskindex = 0;
		base = 0;
		
		while (maskindex < (tmprdfp->oidlist->total/MASK_WORD_SIZE +1)) {
				/* for each long-word mask */
		  mask = Nlm_SwapUint4(tmprdfp->oidlist->list[maskindex]);
		  
		  i = 0;
		  while (mask) {
		    if (mask & (((Uint4)0x1)<<(MASK_WORD_SIZE-1))) {
		      oid = base + i;
		      virtual_oid = oid + tmprdfp->start;
		      
		      virtual_mask_index = virtual_oid/MASK_WORD_SIZE;
		      virtual_oid_bit = 0x1 << (MASK_WORD_SIZE - 1 - virtual_oid % MASK_WORD_SIZE);
		      virtual_oidlist->list[virtual_mask_index] |= virtual_oid_bit;
		    }
		    mask <<= 1;
		    i++;
		  }
		  maskindex++;
		  base += MASK_WORD_SIZE;
		}
		
		/* free old mask */
		tmprdfp->oidlist = OIDListFree(tmprdfp->oidlist);
	      }
	      tmprdfp = tmprdfp->next;
	    
	      if (virtual_oidlist) {
		for (i=0; i<total_virtual_mask; i++) {
		  virtual_oidlist->list[i] = Nlm_SwapUint4(virtual_oidlist->list[i]);
		}
	      }
	      search->rdfp->oidlist = virtual_oidlist;
	      
	      readdb_get_totals_ex(search->rdfp, &(dblen), &(search->dbseq_num), TRUE);
	      
	    }
		/* Intended for use when a list of gi's is sent in, but the real size is needed. */
		/* It's probably still necessary to call BlastAdjustDbNumbers, but it would be nice
			if this were not required. */
		if (options->use_real_db_size)
			readdb_get_totals(search->rdfp, &(dblen), &(search->dbseq_num));

#if 0
		/* use length and num of seqs of the database from alias file */
		if (search->rdfp->aliaslen && !gi_list)
		    dblen = search->rdfp->aliaslen;
		if (search->rdfp->aliasnseq && !gi_list) 
		    search->dbseq_num = search->rdfp->aliasnseq;
#endif
		/* command-line/options trump alias file. */
		if (options->db_length > 0)
			dblen = options->db_length;
		if (options->dbseq_num > 0)
			search->dbseq_num = options->dbseq_num;
		if (options->searchsp_eff > 0)
			searchsp_eff = options->searchsp_eff;

		search->dblen = dblen;
		search->searchsp_eff = searchsp_eff;
		status = BLASTSetUpSearchInternalByLoc (search, slp, query_bsp, blast_program, SeqLocLen(slp), options, tick_callback);
		if (status != 0)
		{
	  		ErrPostEx(SEV_WARNING, 0, 0, "SetUpBlastSearch failed.");
			search->query_invalid = TRUE;
		}
	}



      /* END OF ALTERNATE BLASTSetUpSearchWithReadDbInternal CALL */

      if (search == NULL) {
	printf("ERROR IN BLASTSetUpSearchByLocWithReadDbEx\n");
	return 1;
      }
      search->rdfp = ReadDBCloseMHdrAndSeqFiles(search->rdfp);
      search->thr_info->tick_callback = tick_callback;
      search->thr_info->star_callback = tick_callback;
      seqalign = MyBioseqBlastEngineCore(search, options, NULL);
      if (search->error_return) {
	ValNodeLink(&error_returns, search->error_return);
	search->error_return = NULL;
      }

      /*
      if (other_returns) { 
	other_returns = BlastOtherReturnsPrepare(search);
      }
      */

      search->rdfp = ReadDBCloseMHdrAndSeqFiles(search->rdfp);
      search->rdfp = ReadDBFreeSharedInfo(search->rdfp);
      search = BlastSearchBlkDestruct(search);
      SeqLocFree(slp);
      /* END OF THE ALTERNATE BioseqBlastEngine CALL  */

      /* ADDED THIS TO WRITE OWN OUTPUT TO outfp */
      MyWriteHits(seqalign, query, fb, db, hdrs, outfp); 
      
      BlastErrorPrint(error_returns);
      
      /* THIS STUFF TAKEN OUT FOR NOW... MAY COME BACK LATER
	 dbinfo = NULL;
	 ka_params = NULL;
	 ka_params_gap = NULL;
	 params_buffer = NULL;
	 mask_loc = NULL;
	 for (vnp=other_returns; vnp; vnp = vnp->next)
	 {
	 switch (vnp->choice) {
	 case TXDBINFO:
	 dbinfo = vnp->data.ptrvalue;
	 break;
	 case TXKABLK_NOGAP:
	 ka_params = vnp->data.ptrvalue;
	 break;
	 case TXKABLK_GAP:
	 ka_params_gap = vnp->data.ptrvalue;
	 break;
	 case TXPARAMETERS:
	 params_buffer = vnp->data.ptrvalue;
	 break;
	 case TXMATRIX:
	 break;
	 case SEQLOC_MASKING_NOTSET:
	 case SEQLOC_MASKING_PLUS1:
	 case SEQLOC_MASKING_PLUS2:
	 case SEQLOC_MASKING_PLUS3:
	 case SEQLOC_MASKING_MINUS1:
	 case SEQLOC_MASKING_MINUS2:
	 case SEQLOC_MASKING_MINUS3:
	 ValNodeAddPointer(&mask_loc, vnp->choice, vnp->data.ptrvalue);
	 break;
	 default:
	 break;
	 }
	 }	
	 
	 ReadDBBioseqFetchEnable ("blastall", blast_database, db_is_na, TRUE);
	 
	 printf("Before PrintDbReport \n");
	 
	 init_buff_ex(85);
	 dbinfo_head = dbinfo;
	 while (dbinfo)
	 {
	 PrintDbReport(dbinfo, 70, outfp);
	 dbinfo = dbinfo->next;
	 }
	 dbinfo_head = TxDfDbInfoDestruct(dbinfo_head);
	 
	 if (ka_params) {
	 PrintKAParameters(ka_params->Lambda, ka_params->K, ka_params->H, 70, outfp, FALSE);
	 MemFree(ka_params);
	 }
	 if (ka_params_gap) {
	 PrintKAParameters(ka_params_gap->Lambda, ka_params_gap->K, ka_params_gap->H, 70, outfp, TRUE);
	 MemFree(ka_params_gap);
	 }
	 PrintTildeSepLines(params_buffer, 70, outfp);
	 
	 MemFree(params_buffer);
	 free_buff();
	 mask_loc_start = mask_loc;
	 while (mask_loc)
	 {
	 SeqLocSetFree(mask_loc->data.ptrvalue);
	 mask_loc = mask_loc->next;
	 }
	 ValNodeFree(mask_loc_start);
      */
      
      fake_bsp = BlastDeleteFakeBioseq(fake_bsp);
      
      /* 
	 ReadDBBioseqFetchDisable();
      */
      
      other_returns = ValNodeFree(other_returns);
      sep = SeqEntryFree(sep);
    }
  options = BLASTOptionDelete(options);

  FileClose(infp);

  
  
  /* BELOW THIS IS THE STUFF FROM ncbimain.c */
  /* THIS CODE IS SKIPPED SINCE ITS 'just' CLEAN UP.... 
     NlmThreadJoinAll();
     
     Nlm_FreeConfigStruct();
     ErrSetLogfile(NULL, 0);
     Nlm_ReleaseAppContext();
     
     #ifdef MSC_VIRT
     _vheapterm();
     #endif
     
     NlmThreadDestroyAll();
  */
  return 0;
}
	

