/*-----------------------------------------------------------------------------

   QUASAR - q-gram Alignment based on Suffix ARrays

   Copyright (C) 1998 Stefan Burkhardt
   Author: Stefan Burkhardt <stburk@mpi-sb.mpg.de>
   This file is part of the QUASAR package.

   QUASAR is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   QUASAR is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the QUASAR package; see the file copying.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  or contact the author. 

-------------------------------------------------------------------------------

  utility functions (mem allocation, file handling ...)
  
  $File$
  $Revision: 1.2 $
  $Date: Wed, 29 Mar 2000 11:07:45 +0200 $

-----------------------------------------------------------------------------*/

#include "q_misc.h"

void *MyMalloc(LONG size, char *var_name)
/*  Attempts to allocate size bytes of memory with malloc. On
    success returns pointer to allocated area, if malloc fails
    prints an errormessage including the string var_name and
    exits with OUT_OF_VIRTUAL_MEM (see q_misc.h) */ 
{
  void *mptr;

  mptr = malloc((size_t)size);

  if(mptr == NULL)
    {
      printf("ERROR: out of memory allocating %lld bytes for %s\n", 
	      size, var_name);
      exit(OUT_OF_VIRTUAL_MEM);
    }
#ifdef DEBUG
  printf("Allocated %lld bytes for %s at %d\n", size, var_name, (int)mptr);
  memset(mptr, 0, size);
#endif

  return mptr;
}


void *MyRealloc(void *ptr, LONG size, char *var_name)
/*  Attempts to reallocate ptr to size bytes of memory with
    realloc. On success returns pointer to allocated area, if
    realloc fails prints an errormessage incldung the string
    var_name and exits with OUT_OF_VIRTUAL_MEM (see q_misc.h) */
{
  void *mptr;
  
  mptr = realloc(ptr, (size_t)size);
  if(mptr == NULL)
    {
      printf("ERROR: out of memory reallocating %lld bytes for %s\n", 
              size, var_name);
      exit(OUT_OF_VIRTUAL_MEM);
    }
  return mptr;
}


FILE *MyFopen(char *name, char *mode)
/*  Attempts to open the file name in the mode mode. On success
    returns the file pointer, if fopen fails prints an errormessage 
    including the file name and exits with FOPEN_FAILED. This code
    uses 64 bit file access. It requires compiler support (see makefile)
    and uses the function fopen64 */
{
  FILE *fptr;

  fptr = fopen64(name, mode);
  if(fptr == NULL)
    {
      printf("ERROR: failed to open file %s\n", name);
      exit(FOPEN_FAILED);
    }
  return fptr;
}

void MyFread(void *target, LONG bytes, FILE *infile)
/*  Attempts to read bytes byte from infile to the target. It reads in blocks
    of size FILE_BLOCK. It exits with FREAD_FAILED (see q_misc.h) if less than
    bytes byte have been read */
{
  LONG	i, full_blocks, remainder;

  if(bytes<=0)
    {
      printf("ERROR: %d bytes requested from fread \n", bytes);
      exit(FREAD_FAILED);
    }
  full_blocks = bytes / FILE_BLOCK;
  remainder = bytes % FILE_BLOCK;
#ifdef DEBUG
  printf("Full blocks: %lld, remainder: %lld, total: %lld\n", full_blocks, remainder, bytes);
#endif
  for(i=0; i<full_blocks; i++) {
    if(fread(target, 1, FILE_BLOCK, infile) != FILE_BLOCK) {
      printf("ERROR: failed to read %d bytes from disk (Block %lld)\n", FILE_BLOCK, i);
      exit(FREAD_FAILED);
    }
    (char *)target += FILE_BLOCK;
  }
  if(fread(target, 1, remainder, infile) != (int)remainder) {
    printf("ERROR: failed to read %lld bytes from disk (Remainder)\n", remainder);
    exit(FREAD_FAILED);
  }
}

void MyFwrite(void *source, LONG bytes, FILE *outfile)
/*  Attempts to write bytes byte from the source to outfile. Writes in blocks
    of size FILE_BLOCK. It exits with FWRITE_FAILED (see q_misc.h) if less than
    bytes byte have been written */
{
  LONG	i, full_blocks, remainder;

  if(bytes<=0) {
    printf("ERROR: %d bytes requested from fwrite \n", bytes);
    exit(FWRITE_FAILED);
  }
  full_blocks = bytes / FILE_BLOCK;
  remainder = bytes % FILE_BLOCK;
#ifdef DEBUG
  printf("Full blocks: %lld, remainder: %lld, total: %lld\n", full_blocks, remainder, bytes);
#endif
  for(i=0; i<full_blocks; i++) {
    if(fwrite(source, 1, FILE_BLOCK, outfile) != FILE_BLOCK) {
      printf("ERROR: failed to write %d bytes to disk (Block %lld)\n", FILE_BLOCK, i);
      exit(FWRITE_FAILED);
    }
    (char *)source += FILE_BLOCK;
  }
  if(fwrite(source, 1, remainder, outfile) != (int)remainder) {
    printf("ERROR: failed to write %lld bytes to disk (Remainder)\n", remainder);
    exit(FWRITE_FAILED);
  }
}
